/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
 * File: xsetutil.c 
 *       Helper functions for the setting database
 * -----------------------------------------------------------------*/

#include <assert.h>

#include "xsetutil.h"

/********************************************************************
  Accessing property memories ***************************************
********************************************************************/

/* This function is called from the PropSet() functions for 
   block- and behavior memory. It writes the given
   property value to its location in the setting database.
   It writes exactly one WORD resp. one DW value to memory.
   If a property covers more than a WORD resp. a DW (can be any property 
   with len>2 bit), you have to call this function more often.
   It returns 1 on success.
   Returning 0 is a fatal error and should never happen.
 */

bx_bool BestXMemPropSet
(
  bx_int32 *MemLine,       /* pointer to memory line (depends on handle) */
  bx_int32 FirstBitPos,    /* first bitpos to change (in [0,..,15 resp.31]) */
  bx_int32 LastBitPos,     /* last bitpos to change  (in [0,..,15 resp 31]) */
  bx_int32 PropVal,        /* up to 16 resp 32 bit property value to write  */
  bx_charptrtype PropName, /* for debugging only: the property name */
  bx_int32 RegSize         /* 2 (behavior) or 4 (block) byte accesses */
)
{
  bx_int32 Mask; /* for resetting the property bits in database to zero */

  /* First some parameter checking */
  if (!MemLine||FirstBitPos>LastBitPos||
      RegSize!=2 && RegSize!=4||LastBitPos-FirstBitPos>=RegSize*8)
  {
    /* This is a fatal parameter error */
    return 0;
  }

  /* Just to be sure: remove invalid upper bits from PropVal */
  Mask=0;
  if (!BestXBitMaskModify(&Mask,0,LastBitPos-FirstBitPos,1))
  {
    return 0; /* parameter error, should never happen */
  }
  /* Mask now has ones on valid range [0,..,LastBitPos-FirstBitPos] */
  PropVal &= Mask; /* removes upper, unused, but maybe set bits */


  Mask=~0; /* 32 one's: 111...111\b */
  if (!BestXBitMaskModify(&Mask,FirstBitPos,LastBitPos,0))
  {
    return 0; /* parameter error, should never happen */
  }
  /* Mask now has zeros on property range [FirstBitPos,..,LastBitPos] */
  
  /* write PropVal to *MemLine */
  if (RegSize==2)
  {
    *(bx_int16 *)MemLine &= (bx_int16) Mask;      /* clear bits of property to zero */ 
    *(bx_int16 *)MemLine |= PropVal<<FirstBitPos; /* write property */ 
  }
  else
  {
    assert(RegSize==4);
    *MemLine &=  Mask;                /* clear bits of property to zero */ 
    *MemLine |= PropVal<<FirstBitPos; /* write property */ 
  }

#ifdef _XDEBUG
  printf("Writing: Prop:%s, FirstBitPos: %lu, LastBitPos: %lu val=%lu mem=%08lx\n",PropName,FirstBitPos,LastBitPos,PropVal,*MemLine);
#endif

  return 1; /* OK */
}

/* This function is called from the PropGet() functions for 
   block- and behavior memory. It reads the given
   property value from its location in the setting database.
   It reads exactly one WORD resp. DW value from memory.
   If a property covers more than a WORD resp DW (can be any property 
   with len>2 bit), you have to call this function more often.
   It returns 1 on success.
   Returning 0 is a fatal error and should never happen.
 */

bx_bool BestXMemPropGet
(
  bx_int32 *MemLine,       /* pointer to memory line (depends on handle) */
  bx_int32 FirstBitPos,    /* first bitpos to read (in [0,..,15 resp 32]) */
  bx_int32 LastBitPos,     /* last bitpos to read  (in [0,..,15 resp 32]) */
  bx_int32 *PropVal,       /* up to 16 resp 32 bit property value to read  */
  bx_charptrtype PropName, /* for debugging only: the property name */
  bx_int32 RegSize         /* 2 (behavior) or 4 (block) byte accesses */
)
{
  bx_int32 Mask; /* for getting bit range from database */

  /* First some parameter checking */
  if (!MemLine||FirstBitPos>LastBitPos||
      RegSize!=2 && RegSize!=4||LastBitPos-FirstBitPos>=RegSize*8)
  {
    /* This is a fatal parameter error */
    return 0;
  }

  *PropVal=0;
  Mask=0; 
  if (!BestXBitMaskModify(&Mask,FirstBitPos,LastBitPos,1))
  {
    return 0; /* parameter error, should never happen */
  }
  /* Mask now has ones on property range [FirstBitPos,..,LastBitPos] */
  
  /* read *PropVal from *MemLine;
     Valid bits start at position zero, so we have to shift right */
  
  if (RegSize==2)
  {
    *PropVal = ((bx_int16) Mask & *(bx_int16 *)MemLine) >>  FirstBitPos;
  }
  else
  {
    assert(RegSize==4);
    *PropVal = (Mask & *MemLine) >>  FirstBitPos;
  }

#ifdef _XDEBUG
  printf("Reading: Prop:%s, FirstBitPos: %lu, LastBitPos: %lu val=%lu\n",PropName,FirstBitPos,LastBitPos,*PropVal);
#endif

  return 1; /* OK */
}


/********************************************************************
  Debugging *********************************************************
********************************************************************/

bx_bool EXPORT BestXPrintDB(bx_handletype handle,bx_charptrtype FileName)
{
  FILE *fp;

  if(!FileName||(fp=BESTX_FOPEN(FileName,"a+"))==NULL)
  {
    /* not write to file but to standard output */
    fp=stdout;
  }

  BESTX_FPRINTF(fp,"\n\n");
  BESTX_FPRINTF(fp,"*********************************************\n");
  BESTX_FPRINTF(fp,"********* BESTX Host Database ***************\n");
  BESTX_FPRINTF(fp,"*********************************************\n\n");

#if 1
  BestXPrintMBlkDB(handle,fp,NULL);
  BestXPrintMBehDB(handle,fp,NULL);
  BestXPrintTBehDB(handle,fp,NULL);
  BestXPrintCBehDB(handle,fp,NULL);
  BestXPrintRBehDB(handle,fp,NULL);
  BestXPrintGenPropDB(handle,fp,NULL);
#endif

  BestXPrintConfigDB(handle,fp,NULL);

  fclose(fp);

  return 1;
}

bx_bool EXPORT BestXPrintMBlkDB(bx_handletype handle, FILE *fp,bx_charptrtype Message)
{
  bx_int32 i,j;

  if (!fp)
  {
    fp=stdout;
  }

  BESTX_FPRINTF(fp,"\n\tRI Block (HostDB) (%s)\n",(Message?Message:""));

  for (i=0;i<BX_RIBLK_MEMDEPTH;i++)
  {
    BESTX_FPRINTF(fp,"\n[%3lu]: ",i);
    for (j=0;j<BX_RIBLK_MEMWIDTH;j++)
    {
      BESTX_FPRINTF(fp,"%08x ",bx_handlearray[handle].db->Exerciser.RequesterInitiator.Block.Mem[i][j]);
    }
  }

  return 1;
}


bx_bool EXPORT BestXPrintMBehDB(bx_handletype handle, FILE *fp,bx_charptrtype Message)
{
  bx_int32 i,j;

  if (!fp)
  {
    fp=stdout;
  }

  BESTX_FPRINTF(fp,"\n\tRI Behavior (HostDB) (%s)\n",(Message?Message:""));

  for (i=0;i<BX_RIBEH_MEMDEPTH;i++)
  {
    BESTX_FPRINTF(fp,"\n[%3lu]: ",i);
    for (j=0;j<BX_RIBEH_MEMWIDTH;j++)
    {
      BESTX_FPRINTF(fp,"%04x ",bx_handlearray[handle].db->Exerciser.RequesterInitiator.Behavior.Mem[i][j]);
    }
  }

  return 1;
}

bx_bool EXPORT BestXPrintTBehDB(bx_handletype handle, FILE *fp,bx_charptrtype Message)
{
  bx_int32 i,j;

  if (!fp)
  {
    fp=stdout;
  }

  BESTX_FPRINTF(fp,"\n\tTarget Behavior (HostDB) (%s)\n",(Message?Message:""));

  for (i=0;i<BX_CTBEH_MEMDEPTH;i++)
  {
    BESTX_FPRINTF(fp,"\n[%3lu]: ",i);
    for (j=0;j<BX_CTBEH_MEMWIDTH;j++)
    {
      BESTX_FPRINTF(fp,"%04x ",bx_handlearray[handle].db->Exerciser.CompleterTarget.Behavior.Mem[i][j]);
    }
  }

  return 1;
}

bx_bool EXPORT BestXPrintCBehDB(bx_handletype handle, FILE *fp,bx_charptrtype Message)
{
  bx_int32 i,j;

  if (!fp)
  {
    fp=stdout;
  }

  BESTX_FPRINTF(fp,"\n\tCompleter Behavior (HostDB) (%s)\n",(Message?Message:""));

  for (i=0;i<BX_CIBEH_MEMDEPTH;i++)
  {
    BESTX_FPRINTF(fp,"\n[%3lu]: ",i);
    for (j=0;j<BX_CIBEH_MEMWIDTH;j++)
    {
      BESTX_FPRINTF(fp,"%04x ",bx_handlearray[handle].db->Exerciser.CompleterInitiator.Behavior.Mem[i][j]);
    }
  }

  return 1;
}

bx_bool EXPORT BestXPrintRBehDB(bx_handletype handle, FILE *fp,bx_charptrtype Message)
{
  bx_int32 i,j;

  if (!fp)
  {
    fp=stdout;
  }

  BESTX_FPRINTF(fp,"\n\tRequester Behavior (HostDB) (%s)\n",(Message?Message:""));

  for (i=0;i<BX_RTBEH_MEMDEPTH;i++)
  {
    BESTX_FPRINTF(fp,"\n[%3lu]: ",i);
    for (j=0;j<BX_RTBEH_MEMWIDTH;j++)
    {
      BESTX_FPRINTF(fp,"%04x ",bx_handlearray[handle].db->Exerciser.RequesterTarget.Behavior.Mem[i][j]);
    }
  }

  return 1;
}

bx_bool EXPORT BestXPrintGenPropDB(bx_handletype handle, FILE *fp,bx_charptrtype Message)
{
  bx_int32 i,j;

  if (!fp)
  {
    fp=stdout;
  }

  BESTX_FPRINTF(fp,"\n\tGeneric Properties (HostDB) (%s)\n",(Message?Message:""));

  BESTX_FPRINTF(fp,"\n\tGeneric Exerciser Properties\n");
  for (i=0;i<BX_EGENPROP_SIZE;i++)
  {
    BESTX_FPRINTF(fp,"[%3lu]: ",i);
    BESTX_FPRINTF(fp,"%lu\n",bx_handlearray[handle].db->Exerciser.GenProp[i]);
  }

  BESTX_FPRINTF(fp,"\n\tGeneric RI Properties\n");
  for (i=0;i<BX_RIGENPROP_SIZE;i++)
  {
    BESTX_FPRINTF(fp,"[%3lu]: ",i);
    BESTX_FPRINTF(fp,"%lu\n",bx_handlearray[handle].db->Exerciser.RequesterInitiator.GenProp[i]);
  }

  BESTX_FPRINTF(fp,"\n\tGeneric CT Properties\n");
  for (i=0;i<BX_CTGENPROP_SIZE;i++)
  {
    BESTX_FPRINTF(fp,"[%3lu]: ",i);
    BESTX_FPRINTF(fp,"%lu\n",bx_handlearray[handle].db->Exerciser.CompleterTarget.GenProp[i]);
  }

  BESTX_FPRINTF(fp,"\n\tCT Split Conditions [instance,prop]\n");
  for (i=0;i<BX_CTSPLITPROP_DEPTH;i++)
  {
    for (j=0;j<BX_CTSPLITPROP_SIZE;j++)
    {
      BESTX_FPRINTF(fp,"[%3lu,%3lu]: ",i,j);
      BESTX_FPRINTF(fp,"%lu\n",bx_handlearray[handle].db->Exerciser.CompleterTarget.SplitCondProp[i][j]);
    }
  }

  BESTX_FPRINTF(fp,"\n\tGeneric CI Properties\n");
  for (i=0;i<BX_CIGENPROP_SIZE;i++)
  {
    BESTX_FPRINTF(fp,"[%3lu]: ",i);
    BESTX_FPRINTF(fp,"%lu\n",bx_handlearray[handle].db->Exerciser.CompleterInitiator.GenProp[i]);
  }

  BESTX_FPRINTF(fp,"\n\tGeneric RT Properties\n");
  for (i=0;i<BX_RTGENPROP_SIZE;i++)
  {
    BESTX_FPRINTF(fp,"[%3lu]: ",i);
    BESTX_FPRINTF(fp,"%lu\n",bx_handlearray[handle].db->Exerciser.RequesterTarget.GenProp[i]);
  }

  BESTX_FPRINTF(fp,"\n\tGeneric Analyzer Properties\n");
  for (i=0;i<BX_AGENPROP_SIZE;i++)
  {
    BESTX_FPRINTF(fp,"[%3lu]: ",i);
    BESTX_FPRINTF(fp,"%lu\n",bx_handlearray[handle].db->Analyzer.GenProp[i]);
  }

  return 1;
}

bx_bool EXPORT BestXPrintConfigDB(bx_handletype handle, FILE *fp,bx_charptrtype Message)
{
  bx_int32 i,j;
  bx_int32 ConfigSize;

  if (!fp)
  {
    fp=stdout;
  }

  BESTX_FPRINTF(fp,"\n\tConfig Space (HostDB) (%s)\n",(Message?Message:""));

  ConfigSize=(BestXHasMephisto(handle)?BX_CTCFG_MEMDEPTH:BX_CTCFG_MEMDEPTHMAX);

  for (i=0;i<ConfigSize;i++)
  {
    BESTX_FPRINTF(fp,"\nMask  [%3lu]: ",i);
    for(j=0;j<32;j++)
    {
      BESTX_FPRINTF(fp,"%c",(1<<(31-j) & (bx_handlearray[handle].db->Exerciser.CompleterTarget.ConfigSpace.mask[i])?'1':'0'));
    }

    BESTX_FPRINTF(fp," = 0x%08lx",bx_handlearray[handle].db->Exerciser.CompleterTarget.ConfigSpace.mask[i]);

    BESTX_FPRINTF(fp,"\nValue [%3lu]: ",i);
    for(j=0;j<32;j++)
    {
      BESTX_FPRINTF(fp,"%c",(1<<(31-j) & (bx_handlearray[handle].db->Exerciser.CompleterTarget.ConfigSpace.value[i])?'1':'0'));
    }

    BESTX_FPRINTF(fp," = 0x%08lx",bx_handlearray[handle].db->Exerciser.CompleterTarget.ConfigSpace.value[i]);
  }

  return 1;
}

/********************************************************************
  Misc **************************************************************
********************************************************************/

/* Helper function:
   Sets given range of bits to 0 or 1.
   It returns 1 on success.
   Returning 0 is a wrong parameter range
 */
bx_bool BestXBitMaskModify(
  bx_int32 *Value,          /* <INOUT> */
  bx_int32 FirstBitPos,     /* first bit to modify */
  bx_int32 LastBitPos,      /* last bit to modify */
  bx_int32 BitVal           /* value for bits to set (0 or 1) */ 
)
{
  /* Sets bits FirstBitPos to LastBitPos of *Value to BitVal */

  bx_int32 i;   

  /* Check input parameters */
  if (!Value||FirstBitPos>LastBitPos||BitVal>1||FirstBitPos>31||LastBitPos>31)
  {
    /* Input parameter wrong */
    return 0;  
  }
  
  for (i=FirstBitPos;i<=LastBitPos;i++)
  {
    /* These bits are set to value BitVal  */

    if (BitVal)
    {
      *Value |= 1<<i;    /* set i-th bit   */
    }
    else
    {
      *Value &= ~(1<<i); /* clear i-th bit */
    }
  }

  return 1;
}
